/**************************************************************************//**
 * @file        RTC.c
 * @version     V1.00.03
 * $Revision:   0 $
 * $Date:       2018-04-16 $
 * @brief       RTC TIMER Driver source file
 *
 * @note        V1.00.02: Modify function for robust & correctness.
 *
 * Copyright (C) 2017 AMICCOM Electronics Corp. All rights reserved.
 *****************************************************************************/

#include "A9139M0.h"
#include "Delay.h"
/** @addtogroup Std_Driver Standard Driver
    @{
*/
/** @addtogroup RTC_Driver Real Time Clock Driver
    @{
*/

/**
 *
 * @brief   Initial RTC
 *
 * @param   clksel                  Clock source select. 0=32KHz, 1=32.768KHz
 * @param   year                    Set year,range=0~99
 * @param   month                   Set month, range=1~12
 * @param   data                    Set data, range=1~31
 * @param   week                    Set WEEK, range=0~6
 * @param   hour                    Set HOUR, range=0~23
 * @param   min                     Set MIN,  range=0~59
 * @param   sec                     Set SEC,  range=0~59
 * @return  none
 */
void RTC_Initial(uint32_t clksel,uint32_t week, uint32_t hour, uint32_t min, uint32_t sec)
{
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->CTRL = 0;
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->COUNT = (((week << RTC_COUNT_WEEK_Pos) & RTC_COUNT_WEEK_Msk) |
                  ((hour << RTC_COUNT_HOUR_Pos) & RTC_COUNT_HOUR_Msk) |
                  ((min << RTC_COUNT_MIN_Pos) & RTC_COUNT_MIN_Msk) |
                  ((sec << RTC_COUNT_SEC_Pos) & RTC_COUNT_SEC_Msk)) ;
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->CTRL = (((clksel << RTC_CTRL_CLKSEL_Pos) & RTC_CTRL_CLKSEL_Msk) |
                   RTC_CTRL_EN_Msk);
}
void RTCEX_Initial(uint32_t year ,uint32_t month,uint32_t data)
{
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->RTCEX = (((year << RTC_RTCEX_RTCY_Pos) & RTC_RTCEX_RTCY_Msk) |
									((month << RTC_RTCEX_RTCMT_Pos) & RTC_RTCEX_RTCMT_Msk) |
									((data << RTC_RTCEX_RTCDT_Pos) & RTC_RTCEX_RTCDT_Msk)) ;
}

/**
 *
 * @brief   RTC Clock on, RTC Start counting
 *
 * @return  none
 */
void RTC_Start(void)
{
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->CTRL |= RTC_CTRL_ON_Msk;
}


/**
 *
 * @brief   RTC Clock off, RTC Stop count
 *
 * @return  none
 */
void RTC_Stop(void)
{
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->CTRL &= ~RTC_CTRL_ON_Msk;
}

/**
 *
 * @brief   Modify RTC count value
 *
 * @param   clksel                  Clock source select. 0=32KHz, 1=32.768KHz
 * @param   week                    Set WEEK, range=0~6
 * @param   hour                    Set HOUR, range=0~23
 * @param   min                     Set MIN,  range=0~59
 * @param   sec                     Set SEC,  range=0~59
 * @return  none
 */
void RTC_ModifyCount(uint32_t week, uint32_t hour, uint32_t min, uint32_t sec)
{
    uint32_t control, count;
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    control = RTC->CTRL;
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    count = (((week << RTC_COUNT_WEEK_Pos) & RTC_COUNT_WEEK_Msk) |
             ((hour << RTC_COUNT_HOUR_Pos) & RTC_COUNT_HOUR_Msk) |
             ((min << RTC_COUNT_MIN_Pos) & RTC_COUNT_MIN_Msk) |
             ((sec << RTC_COUNT_SEC_Pos) & RTC_COUNT_SEC_Msk));
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->CTRL &= ~(RTC_CTRL_EN_Msk);
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->COUNT =  count;
    Delay10us(4);                    // wait 1 RTC clock (1/32K = 31.25us)
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->CTRL = control;
}

/**
 *
* @brief   Set Alarm. The Alarm interrupt will occure when RTC->CTRL[hour:min] == RTC->ALARM[hour:min]
 *
 * @param   hour                    Set HOUR, range=0~23
 * @param   min                     Set MIN,  range=0~59
 * @return  none
 */
void RTC_SetAlarm(uint32_t hour, uint32_t min)
{		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->ALARM = (((hour << RTC_ALARM_HOUR_Pos) & RTC_ALARM_HOUR_Msk) |
                  ((min << RTC_ALARM_MIN_Pos) & RTC_ALARM_MIN_Msk));
}

/**
 *
 * @brief   Set Manual Count
 *
 * @param   en                      Menaul Count Enable. 0=disable, 1=enable
 * @param   count                   Menaul Count set
 * @return  none
 */
void RTC_ManualCount_Set(uint32_t en, uint32_t count)
{		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->MANUALCOUNT = (count << RTC_MANUALCOUNT_COUNT_Pos) & RTC_MANUALCOUNT_COUNT_Msk;
    if(0 == en)
    {
				HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
				HW32_REG(0x50005018) = 0x55555555;
        RTC->CTRL &= ~RTC_CTRL_MANUALCOUNTEN_Msk;
    }
    else
    {
				HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
				HW32_REG(0x50005018) = 0x55555555;
        RTC->CTRL |= RTC_CTRL_MANUALCOUNTEN_Msk;
    }
}

/**
 *
 * @brief   Enable RTC interrupt
 *
 * @param   alarm                   Alarm interrupt.        0=disable, 1=enable.
 * @param   sec                     Sec interrupt.          0=disable, 1=enable.
 * @param   half_sec                Half sec interupt.      0=disable, 1=enable.
 * @param   ten_ms                  Ten ms interrupt.       0=disable, 1=enable.
 * @param   manual                  Manaul Count interrupt. 0=disable, 1=enable.
 * @return  none
 */
void RTC_Interrupt_Enable(uint32_t alarm, uint32_t sec, uint32_t half_sec, uint32_t ten_ms, uint32_t manual_count)
{
		
    uint32_t control;
		volatile uint32_t status;
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
	control = RTC->CTRL & (~(RTC_CTRL_MANUALCOUNTIE_Msk | RTC_CTRL_TMSIE_Msk | RTC_CTRL_HSECIE_Msk | RTC_CTRL_SECIE_Msk | RTC_CTRL_ALARMIE_Msk));
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    control |= (((manual_count << RTC_CTRL_MANUALCOUNTIE_Pos) & RTC_CTRL_MANUALCOUNTIE_Msk) |
                ((ten_ms << RTC_CTRL_TMSIE_Pos) & RTC_CTRL_TMSIE_Msk) |
                ((half_sec << RTC_CTRL_HSECIE_Pos) & RTC_CTRL_HSECIE_Msk) |
                ((sec << RTC_CTRL_SECIE_Pos) & RTC_CTRL_SECIE_Msk) |
                ((alarm << RTC_CTRL_ALARMIE_Pos) & RTC_CTRL_ALARMIE_Msk));
		HW32_REG(0x50005018) = 0xAAAAAAAA;  //Protect
	  HW32_REG(0x50005018) = 0x55555555;
    RTC->CTRL = control;
		status = (RTC->CTRL & 0xFFFF);
}


/*@}*/ /* end of group RTC_Driver */
/*@}*/ /* end of group Std_Driver */
